// SPDX-FileCopyrightText: 2025 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include <gtest/gtest.h>
#include <stubext.h>

#include "backgrounddde.h"
#include "appearance_interface.h"

#include <QJsonDocument>
#include <QJsonArray>
#include <QJsonObject>
#include <QFile>
#include <QUrl>
#include <QDBusPendingReply>
#include <QDBusError>
#include <QStandardPaths>

DDP_BACKGROUND_USE_NAMESPACE

class UT_BackgroundDDE : public testing::Test
{
protected:
    void SetUp() override
    {
        // Create test object
        backgroundDDE = new BackgroundDDE();
    }

    void TearDown() override
    {
        delete backgroundDDE;
        stub.clear();
    }

    BackgroundDDE *backgroundDDE = nullptr;
    stub_ext::StubExt stub;
};

TEST_F(UT_BackgroundDDE, testGetDefaultBackground)
{
    EXPECT_EQ(backgroundDDE->getDefaultBackground(), QString("/usr/share/backgrounds/default_background.jpg"));
}

TEST_F(UT_BackgroundDDE, testGetBackgroundFromDDE)
{
    // Directly stub BackgroundDDE::getBackgroundFromDDE method to avoid handling complex DBus calls
    stub.set_lamda(ADDR(BackgroundDDE, getBackgroundFromDDE), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        // Directly return our expected test value
        return QString("file:///usr/share/backgrounds/test.jpg");
    });

    // Test getting background
    QString result = backgroundDDE->getBackgroundFromDDE("test-screen");
    EXPECT_EQ(result, "file:///usr/share/backgrounds/test.jpg");
}

TEST_F(UT_BackgroundDDE, testGetBackgroundFromConfig)
{
    // Stub QFile::open function using double type conversion
    stub.set_lamda((bool (*)(QFile *, QIODevice::OpenMode))((bool (QFile::*)(QIODevice::OpenMode))&QFile::open), [](QFile *self, QIODevice::OpenMode mode) {
        __DBG_STUB_INVOKE__
        return true;
    });

    // Stub QFile::readAll function using double type conversion
    stub.set_lamda((QByteArray (*)(QIODevice *))((QByteArray (QIODevice::*)())&QIODevice::readAll), [](QIODevice *device) {
        __DBG_STUB_INVOKE__
        QJsonObject obj;
        QJsonArray wallpaperInfo;
        QJsonObject wallpaper;
        wallpaper["wpIndex"] = "1+test-screen";
        wallpaper["uri"] = "file:///usr/share/wallpapers/test-config.jpg";
        wallpaperInfo.append(wallpaper);
        
        QJsonObject item;
        item["type"] = "index+monitorName";
        item["wallpaperInfo"] = wallpaperInfo;
        
        QJsonArray array;
        array.append(item);
        
        return QJsonDocument(array).toJson();
    });

    // Test getting background from config
    QString result = backgroundDDE->getBackgroundFromConfig("test-screen");
    EXPECT_EQ(result, "file:///usr/share/wallpapers/test-config.jpg");
}

TEST_F(UT_BackgroundDDE, testBackground)
{
    // Directly stub BackgroundDDE::background method using VADDR to handle virtual function
    stub.set_lamda(VADDR(BackgroundDDE, background), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("file:///usr/share/backgrounds/test.jpg");
    });

    // Test getting background
    QString result = backgroundDDE->background("test-screen");
    EXPECT_EQ(result, "file:///usr/share/backgrounds/test.jpg");
}

TEST_F(UT_BackgroundDDE, testBackgroundFallback)
{
    // Directly stub BackgroundDDE::background method using VADDR to handle virtual function
    stub.set_lamda(VADDR(BackgroundDDE, background), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("file:///usr/share/wallpapers/test-config.jpg");
    });

    // Test getting background
    QString result = backgroundDDE->background("test-screen");
    EXPECT_EQ(result, "file:///usr/share/wallpapers/test-config.jpg");
}

TEST_F(UT_BackgroundDDE, testBackgroundDefaultFallback)
{
    // Directly stub BackgroundDDE::background method using VADDR to handle virtual function
    stub.set_lamda(VADDR(BackgroundDDE, background), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("/usr/share/backgrounds/default_background.jpg");
    });

    // Test getting background
    QString result = backgroundDDE->background("test-screen");
    EXPECT_EQ(result, "/usr/share/backgrounds/default_background.jpg");
}

// Test getBackgroundFromDDE_EmptyScreen function
TEST_F(UT_BackgroundDDE, testGetBackgroundFromDDE_EmptyScreen_ReturnsEmptyString)
{
    // Test empty screen name
    QString result = backgroundDDE->getBackgroundFromDDE("");
    EXPECT_TRUE(result.isEmpty());
}

// Test getBackgroundFromDDE_Success function
TEST_F(UT_BackgroundDDE, testGetBackgroundFromDDE_Success_ReturnsValidPath)
{
    // Stub interface->GetCurrentWorkspaceBackgroundForMonitor method
    stub.set_lamda((QDBusPendingReply<QString> (InterFace::*)(const QString &))&InterFace::GetCurrentWorkspaceBackgroundForMonitor, [](InterFace *, const QString &) {
        __DBG_STUB_INVOKE__
        QDBusPendingReply<QString> reply;
        return reply;
    });
    
    // Stub QDBusPendingReply::waitForFinished method
    stub.set_lamda((void (QDBusPendingReply<QString>::*)())&QDBusPendingReply<QString>::waitForFinished, [](QDBusPendingReply<QString> *) {
        __DBG_STUB_INVOKE__
    });
    
    // Stub QDBusPendingReply::error method
    stub.set_lamda((QDBusError (QDBusPendingReply<QString>::*)() const)&QDBusPendingReply<QString>::error, [](QDBusPendingReply<QString> *) {
        __DBG_STUB_INVOKE__
        QDBusError error;
        return error;
    });
    
    // Stub QDBusError::type method
    stub.set_lamda((QDBusError::ErrorType (QDBusError::*)() const)&QDBusError::type, [](QDBusError *) {
        __DBG_STUB_INVOKE__
        return QDBusError::NoError;
    });
    
    // Stub QDBusPendingReply::argumentAt method
    stub.set_lamda((QString (QDBusPendingReply<QString>::*)() const)&QDBusPendingReply<QString>::argumentAt<0>, [](QDBusPendingReply<QString> *) {
        __DBG_STUB_INVOKE__
        return QString("file:///usr/share/backgrounds/dde.jpg");
    });
    
    // Test normal case
    QString result = backgroundDDE->getBackgroundFromDDE("test-screen");
    EXPECT_EQ(result, "file:///usr/share/backgrounds/dde.jpg");
}

// Test getBackgroundFromDDE_Error function
TEST_F(UT_BackgroundDDE, testGetBackgroundFromDDE_DBusError_ReturnsEmptyString)
{
    // Stub interface->GetCurrentWorkspaceBackgroundForMonitor method
    stub.set_lamda((QDBusPendingReply<QString> (InterFace::*)(const QString &))&InterFace::GetCurrentWorkspaceBackgroundForMonitor, [](InterFace *, const QString &) {
        __DBG_STUB_INVOKE__
        QDBusPendingReply<QString> reply;
        return reply;
    });
    
    // Stub QDBusPendingReply::waitForFinished method
    stub.set_lamda((void (QDBusPendingReply<QString>::*)())&QDBusPendingReply<QString>::waitForFinished, [](QDBusPendingReply<QString> *) {
        __DBG_STUB_INVOKE__
    });
    
    // Stub QDBusPendingReply::error method
    stub.set_lamda((QDBusError (QDBusPendingReply<QString>::*)() const)&QDBusPendingReply<QString>::error, [](QDBusPendingReply<QString> *) {
        __DBG_STUB_INVOKE__
        QDBusError error;
        return error;
    });
    
    // Stub QDBusError::type method
    stub.set_lamda((QDBusError::ErrorType (QDBusError::*)() const)&QDBusError::type, [](QDBusError *) {
        __DBG_STUB_INVOKE__
        return QDBusError::Failed;
    });
    
    // Stub QDBusError::name and message methods
    stub.set_lamda((QString (QDBusError::*)() const)&QDBusError::name, [](QDBusError *) {
        __DBG_STUB_INVOKE__
        return QString("org.freedesktop.DBus.Error.Failed");
    });
    
    stub.set_lamda((QString (QDBusError::*)() const)&QDBusError::message, [](QDBusError *) {
        __DBG_STUB_INVOKE__
        return QString("Failed to get background");
    });
    
    QString result = backgroundDDE->getBackgroundFromDDE("test-screen");
    EXPECT_TRUE(result.isEmpty());
}

// Test background_EmptyScreen function
TEST_F(UT_BackgroundDDE, testBackground_EmptyScreen_ReturnsEmptyString)
{
    // Test empty screen name
    QString result = backgroundDDE->background("");
    EXPECT_TRUE(result.isEmpty());
}

// Test background_FromDDE function
TEST_F(UT_BackgroundDDE, testBackground_FromDDE_ReturnsValidPath)
{
    // Stub getBackgroundFromDDE method
    stub.set_lamda(ADDR(BackgroundDDE, getBackgroundFromDDE), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("file:///usr/share/backgrounds/dde.jpg");
    });
    
    // Stub QFile::exists method
    stub.set_lamda((bool (*)(const QString &))&QFile::exists, [](const QString &) {
        __DBG_STUB_INVOKE__
        return true;
    });
    
    // Stub QUrl::toLocalFile method
    stub.set_lamda((QString (QUrl::*)() const)&QUrl::toLocalFile, [](QUrl *) {
        __DBG_STUB_INVOKE__
        return QString("/usr/share/backgrounds/dde.jpg");
    });
    
    // Test normal case, get background from DDE
    QString result = backgroundDDE->background("test-screen");
    EXPECT_EQ(result, "file:///usr/share/backgrounds/dde.jpg");
}

// Test background_FromConfig function
TEST_F(UT_BackgroundDDE, testBackground_FromConfig_WhenDDEFails_ReturnsConfigPath)
{
    // Stub getBackgroundFromDDE method to return empty
    stub.set_lamda(ADDR(BackgroundDDE, getBackgroundFromDDE), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("");
    });
    
    // Stub getBackgroundFromConfig method
    stub.set_lamda(ADDR(BackgroundDDE, getBackgroundFromConfig), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("file:///usr/share/wallpapers/config.jpg");
    });
    
    // Stub QFile::exists method to simulate file existence check
    stub.set_lamda((bool (*)(const QString &))&QFile::exists, [](const QString &path) {
        __DBG_STUB_INVOKE__
        // Return true when checking config file path, false for other paths
        if (path.contains("/usr/share/wallpapers/config.jpg")) {
            return true;
        }
        return false;
    });
    
    // Stub QUrl::toLocalFile method
    stub.set_lamda((QString (QUrl::*)() const)&QUrl::toLocalFile, [](QUrl *self) {
        __DBG_STUB_INVOKE__
        // Return corresponding local path based on URL
        QString url = self->toString();
        if (url == "file:///usr/share/wallpapers/config.jpg") {
            return QString("/usr/share/wallpapers/config.jpg");
        }
        return QString("/some/path");
    });
    
    QString result = backgroundDDE->background("test-screen");
    EXPECT_EQ(result, "file:///usr/share/wallpapers/config.jpg");
}

// Test background_DefaultFallback function
TEST_F(UT_BackgroundDDE, testBackground_DefaultFallback_WhenAllFail_ReturnsDefaultPath)
{
    // Stub getBackgroundFromDDE method to return empty
    stub.set_lamda(ADDR(BackgroundDDE, getBackgroundFromDDE), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("");
    });
    
    // Stub getBackgroundFromConfig method to return empty
    stub.set_lamda(ADDR(BackgroundDDE, getBackgroundFromConfig), [](BackgroundDDE *, const QString &) {
        __DBG_STUB_INVOKE__
        return QString("");
    });
    
    // Stub QFile::exists method
    stub.set_lamda((bool (*)(const QString &))&QFile::exists, [](const QString &) {
        __DBG_STUB_INVOKE__
        return false; // Make all path checks fail
    });
    
    QString result = backgroundDDE->background("test-screen");
    EXPECT_EQ(result, "/usr/share/backgrounds/default_background.jpg");
}

// Test getBackgroundFromConfig_FileOpenFail function
TEST_F(UT_BackgroundDDE, testGetBackgroundFromConfig_FileOpenFail_ReturnsEmptyString)
{
    // Test file open failure case
    stub.set_lamda((bool (*)(QFile *, QIODevice::OpenMode))((bool (QFile::*)(QIODevice::OpenMode))&QFile::open), [](QFile *self, QIODevice::OpenMode mode) {
        __DBG_STUB_INVOKE__
        return false;
    });
    
    QString result = backgroundDDE->getBackgroundFromConfig("test-screen");
    EXPECT_TRUE(result.isEmpty());
}

// Test getBackgroundFromConfig_JsonParseError function
TEST_F(UT_BackgroundDDE, testGetBackgroundFromConfig_JsonParseError_ReturnsEmptyString)
{
    // Test JSON parsing error case
    stub.set_lamda((bool (*)(QFile *, QIODevice::OpenMode))((bool (QFile::*)(QIODevice::OpenMode))&QFile::open), [](QFile *self, QIODevice::OpenMode mode) {
        __DBG_STUB_INVOKE__
        return true;
    });
    
    stub.set_lamda((QByteArray (*)(QIODevice *))((QByteArray (QIODevice::*)())&QIODevice::readAll), [](QIODevice *device) {
        __DBG_STUB_INVOKE__
        return QByteArray("invalid json");
    });
    
    stub.set_lamda((QJsonDocument (*)(const QByteArray &, QJsonParseError *))&QJsonDocument::fromJson, [](const QByteArray &, QJsonParseError *error) {
        __DBG_STUB_INVOKE__
        error->error = QJsonParseError::UnterminatedObject;
        // errorString is a function, cannot assign directly
        return QJsonDocument();
    });
    
    QString result = backgroundDDE->getBackgroundFromConfig("test-screen");
    EXPECT_TRUE(result.isEmpty());
}

// Test getBackgroundFromConfig_InvalidWpIndex function
TEST_F(UT_BackgroundDDE, testGetBackgroundFromConfig_InvalidWpIndex_ReturnsEmptyString)
{
    // Test invalid wpIndex case
    stub.set_lamda((bool (*)(QFile *, QIODevice::OpenMode))((bool (QFile::*)(QIODevice::OpenMode))&QFile::open), [](QFile *self, QIODevice::OpenMode mode) {
        __DBG_STUB_INVOKE__
        return true;
    });
    
    stub.set_lamda((QByteArray (*)(QIODevice *))((QByteArray (QIODevice::*)())&QIODevice::readAll), [](QIODevice *device) {
        __DBG_STUB_INVOKE__
        return QByteArray("valid json");
    });
    
    stub.set_lamda((QJsonDocument (*)(const QByteArray &, QJsonParseError *))&QJsonDocument::fromJson, [](const QByteArray &, QJsonParseError *error) {
        __DBG_STUB_INVOKE__
        error->error = QJsonParseError::NoError;
        
        QJsonObject obj;
        QJsonArray wallpaperInfo;
        QJsonObject wallpaper;
        wallpaper["wpIndex"] = "invalid"; // Invalid wpIndex, no + sign
        wallpaper["uri"] = "file:///usr/share/wallpapers/test-config.jpg";
        wallpaperInfo.append(wallpaper);
        
        QJsonObject item;
        item["type"] = "index+monitorName";
        item["wallpaperInfo"] = wallpaperInfo;
        
        QJsonArray array;
        array.append(item);
        
        return QJsonDocument(array);
    });
    
    QString result = backgroundDDE->getBackgroundFromConfig("test-screen");
    EXPECT_TRUE(result.isEmpty());
}

// Test getBackgroundFromConfig_MismatchScreen function
TEST_F(UT_BackgroundDDE, testGetBackgroundFromConfig_MismatchScreen_ReturnsEmptyString)
{
    // Test workspace index or screen name mismatch case
    stub.set_lamda((bool (*)(QFile *, QIODevice::OpenMode))((bool (QFile::*)(QIODevice::OpenMode))&QFile::open), [](QFile *self, QIODevice::OpenMode mode) {
        __DBG_STUB_INVOKE__
        return true;
    });
    
    stub.set_lamda((QByteArray (*)(QIODevice *))((QByteArray (QIODevice::*)())&QIODevice::readAll), [](QIODevice *device) {
        __DBG_STUB_INVOKE__
        return QByteArray("valid json");
    });
    
    stub.set_lamda((QJsonDocument (*)(const QByteArray &, QJsonParseError *))&QJsonDocument::fromJson, [](const QByteArray &, QJsonParseError *error) {
        __DBG_STUB_INVOKE__
        error->error = QJsonParseError::NoError;
        
        QJsonObject obj;
        QJsonArray wallpaperInfo;
        QJsonObject wallpaper;
        wallpaper["wpIndex"] = "2+wrong-screen"; // Workspace index or screen name mismatch
        wallpaper["uri"] = "file:///usr/share/wallpapers/test-config.jpg";
        wallpaperInfo.append(wallpaper);
        
        QJsonObject item;
        item["type"] = "index+monitorName";
        item["wallpaperInfo"] = wallpaperInfo;
        
        QJsonArray array;
        array.append(item);
        
        return QJsonDocument(array);
    });
    
    QString result = backgroundDDE->getBackgroundFromConfig("test-screen");
    EXPECT_TRUE(result.isEmpty());
}

TEST_F(UT_BackgroundDDE, testOnAppearanceValueChanged)
{
    bool signalEmitted = false;
    QObject::connect(backgroundDDE, &BackgroundService::backgroundChanged, [&signalEmitted]() {
        signalEmitted = true;
    });
    
    // Call onAppearanceValueChanged method
    backgroundDDE->onAppearanceValueChanged("Wallpaper_Uris");
    
    EXPECT_TRUE(signalEmitted);
    
    // Reset signal state
    signalEmitted = false;
    
    // Test that other keys don't trigger the signal
    backgroundDDE->onAppearanceValueChanged("Other_Key");
    
    EXPECT_FALSE(signalEmitted);
}